/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.ComponentModel;
using System.Reflection;
using Borland.Eco.Handles;
using Borland.Eco.Services;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.UmlRt;

namespace Borland.Eco.Handles
{
	public interface IMaxLengthSupportProvider
	{
		void SetMaxLength(Object obj);
	}

	[AttributeUsage(AttributeTargets.Assembly)]
	public sealed class MaxLengthSupportLibraryAttribute : System.Attribute
	{
		public MaxLengthSupportLibraryAttribute() {}
	}
	
	[AttributeUsage(AttributeTargets.Class)]
	public sealed class MaxLengthSupportProviderAttribute : System.Attribute
	{
		public readonly Type type;
		public MaxLengthSupportProviderAttribute(Type type)
		{
			System.Diagnostics.Debug.Assert(type != null, "type"); // Do not localize
			this.type = type;
		}
	}

	///<summary>
	///This class provides some useful routines to set the max entry length on UI controls based
	///on the modeled max length of (string) attributes).
	///</summary>
	public sealed class MaxLengthSupportFunctions
	{
		private MaxLengthSupportFunctions() {}
		private static System.Collections.Hashtable MaxLengthHashtable;

		///<summary>
		///GetMaxLength returns the modeled length of the attribute pointed to by applying expression
		///to the StaticUmlType of the handle.
		///If the resulting element is not an IAttribute, def is returned.
		///</summary>
		///<param name="expression">An OCL expression applied to the handle's StaticUmlType.</param>
		///<param name="handle">Acting as an IEcoServiceProvider to get to IOclService, and provides the starting type to which to apply expression.</param>
		///<param name="def">Default value returned if the resulting type afer applying expression to the handle's type is not an IAttribute.</param>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="handle"/> is null.</exception>
		public static int GetMaxLength(string expression, ElementHandle handle, int def)
		{
			if (handle == null) 
				throw new ArgumentNullException("handle"); // Do not localize
			IOclService oclservice = (handle as IEcoServiceProvider).GetEcoService(typeof(IOclService)) as IOclService;
			IClassifier context = (handle as IStaticContext).StaticUmlType;
			ICollectionType collectionType = context as ICollectionType;
			if (collectionType != null)
				context = collectionType.ElementType;
			OclVariableCollection variables = (handle.Variables == null) ? null : handle.Variables.OclVariableCollection;
			IStructuralFeature feature = oclservice.ExpressionModelInfo(expression, context, false, variables);
			if (feature is IAttribute)
			{
				int res = (feature as IAttribute).EcoAttribute.Length;
				if (res > 0)
					return res;
			}
			return def;
		}
		///<summary>
		///Returns the GetMaxLength by using the expression of prop.
		///</summary>
		///<param name="prop">The property descriptor used to obtain an expression.</param>
		///<param name="handle">Acting as an IEcoServiceProvider to get to IOclService, and provides the starting type to which to apply expression.</param>
		///<param name="def">Default value returned if prop is null or prop.IsEventDerived is true.</param>
		public static int GetMaxLength(RenderedTuplePropertyDescriptor prop, ElementHandle handle, int def)
		{
			System.Diagnostics.Debug.Assert(handle != null, "handle"); // Do not localize
			if (prop != null && !prop.IsEventDerived)
				return GetMaxLength(prop.Expression, handle, def);
			return def;
		}

		private static void CreateMaxLengthHashtable()
		{
			MaxLengthHashtable = new System.Collections.Hashtable();
			foreach (Assembly assembly in AppDomain.CurrentDomain.GetAssemblies())
			{
				if (assembly.GetCustomAttributes(typeof(MaxLengthSupportLibraryAttribute), true).Length > 0)
				{
					foreach (Type type in assembly.GetTypes())
					{
						object[] maxLengthSupportProviderAttributes;
						maxLengthSupportProviderAttributes = type.GetCustomAttributes(typeof(MaxLengthSupportProviderAttribute), false);
						if (maxLengthSupportProviderAttributes.Length > 0)
						{
							ConstructorInfo constructor = type.GetConstructor(new Type[] {});
							System.Diagnostics.Debug.Assert(constructor != null, "No parameterless constructor found for " + type.Name);
							IMaxLengthSupportProvider maxLengthSupportProvider = (IMaxLengthSupportProvider) constructor.Invoke(new object[] {});
							foreach(MaxLengthSupportProviderAttribute maxLengthSupportProviderAttribute in maxLengthSupportProviderAttributes)
								MaxLengthHashtable[maxLengthSupportProviderAttribute.type] = maxLengthSupportProvider;
						}//length > 0
					}//foreach (type)
				}//MaxLengthSupportLibrary
			}//foreach(Assembly)
		}

		///<summary>
		///Invokes IMaxLengthSupportProvider.SetMaxLength on each object in objects.
		///</summary>
		///<param name="objects">A container of objects that will get their max length set (like a ControlCollection).</param>
		public static void UpdateMaxLengths(IEnumerable objects)
		{
			if (objects == null) return;
			if (MaxLengthHashtable == null)
				CreateMaxLengthHashtable();
			foreach (object obj in objects)
			{
				if (MaxLengthHashtable.Contains(obj.GetType()))
				{
					IMaxLengthSupportProvider maxLengthSupportProvider;
					maxLengthSupportProvider = (IMaxLengthSupportProvider) MaxLengthHashtable[obj.GetType()];
					maxLengthSupportProvider.SetMaxLength(obj);
				}
				UpdateMaxLengths(obj);
			}
		}

		///<summary>
		///Extracts the property Controls from obj (which quite reasonably is a form) using reflection.
		///If there is no such property, nothing is done, no exception is raised.
		///</summary>
		///<param name="obj">A container of objects that will get their max length set (like a form).</param>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		public static void UpdateMaxLengths(object obj)
		{
			if (obj == null) throw new ArgumentNullException("obj"); // Do not localize
			PropertyInfo propinfo = obj.GetType().GetProperty("Controls");
			if (propinfo != null)
			{
				IEnumerable objects = propinfo.GetValue(obj, new object[0]{}) as IEnumerable;
				if (objects != null)
					UpdateMaxLengths(objects);
			}
		}
	}
}
